definition module StdPicture


//	********************************************************************************
//	Clean Standard Object I/O library, version 1.0.1
//	
//	StdPicture contains the drawing operations and access to Pictures.
//	********************************************************************************


from	ospicture	import	Picture
import	StdPictureDef


//	Attribute functions.

getPicture				::						!*Picture -> (!Picture,!*Picture)
getPictureAttributes	::						! Picture -> [PictureAttribute]
setPictureAttributes	:: ![PictureAttribute]	!*Picture -> *Picture

//	Pen position attributes:
setPenPos				:: !Point				!*Picture -> *Picture
getPenPos				::						! Picture -> Point

class movePenPos figure	:: !figure				!*Picture -> *Picture
//	Move the pen position as much as when drawing the figure.
instance movePenPos Vector
instance movePenPos Curve

//	PenSize attributes:
setPenSize				:: !Int					!*Picture -> *Picture 	
getPenSize				::						! Picture -> Int

setDefaultPenSize		::						!*Picture -> *Picture
//	setDefaultPenSize = setPenSize 1

//	Colour attributes:
setPenColour			:: !Colour				!*Picture -> *Picture
getPenColour			::						! Picture -> Colour

setDefaultPenColour		::						!*Picture -> *Picture
//	setDefaultPenColour = setPenColour BlackColour

//	Font attributes:
setPenFont				:: !Font				!*Picture -> *Picture
getPenFont				::						! Picture -> Font

setDefaultPenFont		::						!*Picture -> *Picture

/*	Picture is an environment instance of the FontEnv class.	*/
instance FontEnv Picture


//	Drawing functions.

::	DrawFunction
	:==	*Picture -> *Picture

drawPicture		:: ![DrawFunction] !*Picture -> *Picture
drawseqPicture	:: ![DrawFunction] !*Picture -> *Picture
xorPicture		:: ![DrawFunction] !*Picture -> *Picture
xorseqPicture	:: ![DrawFunction] !*Picture -> *Picture
/*	draw(seq)Picture
		applies the list of drawing functions to the argument picture in left to
		right order. 
		After drawing, drawPicture restores the picture attributes of the resulting 
		picture to those of the argument picture.
	xor(seq)Picture
		applies the list of drawing functions to the argument picture in left to 
		right order in the appropriate platform xor mode.
		After drawing, xorPicture restores the picture attributes of the resulting 
		picture to those of the argument picture. 
*/


//	Drawing points:

drawPoint		::               !*Picture -> *Picture
drawPointAt		:: !Point        !*Picture -> *Picture
/*	drawPoint
		plots a point at the current  pen position p and moves to p+{vx=1,vy=0}
	drawPointAt
		plots a point at the argument pen position, but retains the pen position.
*/

//	Drawing lines:

drawLineTo		:: !Point        !*Picture -> *Picture
drawLine		:: !Point !Point !*Picture -> *Picture
/*	drawLineTo
		draws a line from the current pen position to the argument point which
		becomes the new pen position.
	drawLine
		draws a line between the two argument points, but retains the pen position.
*/


//	Hiliting figures:

class Hilites figure where
	hilite	::        !figure !*Picture -> *Picture
	hiliteAt:: !Point !figure !*Picture -> *Picture
/*	hilite
		draws figures in the appropriate 'hilite' mode at the current  pen position.
	hiliteAt
		draws figures in the appropriate 'hilite' mode at the argument pen position.
	Both functions reset the 'hilite' mode after drawing.
*/

instance Hilites Box		// Hilite a box
instance Hilites Rectangle	// Hilite a rectangle (note: hiliteAt pos r = hilite r)


//	Drawing within in a (list of) clipping area(s):

class Clips area where
	clip	::        !area ![DrawFunction]	!*Picture -> *Picture
	clipAt	:: !Point !area ![DrawFunction]	!*Picture -> *Picture
/*	clip
		takes the base point of the clipping area to be the current  pen position.
	clipAt
		takes the base point of the clipping area to be the argument pen position.
*/

instance Clips Box						// Clip within a box
instance Clips Rectangle				// Clip within a rectangle
instance Clips Polygon					// Clip within a polygon
instance Clips [figure] | Clips figure	// Clip within the union of figures


/*	Drawing and filling operations.
	These functions are divided into the following classes:
	Drawables:	draw     'line-oriented' figures at the current  pen position.
				drawAt   'line-oriented' figures at the argument pen position.
	Fillables:	fill     'area-oriented' figures at the current  pen position.
				fillAt   'area-oriented' figures at the argument pen position.
*/
class Drawables figure where
	draw	::        !figure !*Picture -> *Picture
	drawAt	:: !Point !figure !*Picture -> *Picture

class Fillables figure where
	fill	::        !figure !*Picture -> *Picture
	fillAt	:: !Point !figure !*Picture -> *Picture


//	Text drawing operations:
//	Text is always drawn with the baseline at the y coordinate of the pen.

instance Drawables	Char
instance Drawables	{#Char}
/*	draw     text:
		draws the text starting at the current pen position.
		The new pen position is directly after the drawn text including spacing.
	drawAt p text:
		draws the text starting at p.
*/


//	Vector drawing operations:
instance Drawables Vector
/*	draw     v:
		draws a line from the current pen position pen to pen+v.
	drawAt p v:
		draws a line from p to p+v.
*/


/*	Oval drawing operations:
	An Oval o is a transformed unit circle 
	with	horizontal radius rx	o.oval_rx
			vertical   radius ry	o.oval_ry
	Let (x,y) be a point on the unit circle:
		then (x`,y`) = (x*rx,y*ry) is a point on o.
	Let (x,y) be a point on o:
		then (x`,y`) = (x/rx,y/ry) is a point on the unit circle.
*/
instance Drawables Oval
instance Fillables Oval
/*	draw     o:
		draws an oval with the current pen position being the center of the oval.
	drawAt p o:
		draws an oval with p being the center of the oval.
	fill     o:
		fills an oval with the current pen position being the center of the oval.
	fillAt p o:
		fills an oval with p being the center of the oval.
	None of these functions change the pen position.
*/


/*	Curve drawing operations:
	A Curve c is a slice of an oval o
	with	start angle	a	c.curve_from
			end   angle	b	c.curve_to
			direction   d	c.curve_clockwise
	The angles are taken in radians (counter-clockwise). 
	If d holds then the drawing direction is clockwise, otherwise drawing occurs 
	counter-clockwise.
*/
instance Drawables Curve
instance Fillables Curve
/*	draw     c:
		draws a curve with the starting angle a at the current pen position.
		The pen position ends at ending angle b.
	drawAt p c:
		draws a curve with the starting angle a at p.
	fill     c:
		fills the figure obtained by connecting the endpoints of the drawn curve 
		(draw c) with the center of the curve oval.
		The pen position ends at ending angle b.
	fillAt p c:
		fills the figure obtained by connecting the endpoints of the drawn curve 
		(drawAt p c) with the center of the curve oval.
*/


/*	Box drawing operations:
	A Box b is a horizontally oriented rectangle
	with	width  w		b.box_w
			height h		b.box_h
	In case w==0 (h==0),   the Box collapses to a vertical (horizontal) vector.
	In case w==0 and h==0, the Box collapses to a point.
*/
instance Drawables	Box
instance Fillables	Box
/*	draw     b:
		draws a box with left-top corner at the current pen position p and 
		right-bottom corner at p+(w,h).
	drawAt p b:
		draws a box with left-top corner at p and right-bottom corner at p+(w,h).
	fill     b:
		fills a box with left-top corner at the current pen position p and 
		right-bottom corner at p+(w,h).
	fillAt p b:
		fills a box with left-top corner at p and right-bottom corner at p+(w,h).
	None of these functions change the pen position.
*/


/*	Rectangle drawing operations:
	A Rectangle r is always horizontally oriented
	with	width  w		abs (r.corner1.x-r.corner2.x)
			height h		abs (r.corner1.y-r.corner2.y)
	In case w==0 (h==0),   the Rectangle collapses to a vertical (horizontal) vector.
	In case w==0 and h==0, the Rectangle collapses to a point.
*/
instance Drawables Rectangle
instance Fillables Rectangle
/*	draw     r:
		draws a rectangle with diagonal corners r.corner1 and r.corner2.
	drawAt p r:
		draw r
	fill     r:
		fills a rectangle with diagonal corners r.corner1 and r.corner2.
	fillAt p r:
		fill r
	None of these functions change the pen position.
*/


/*	Polygon drawing operations:
	A Polygon p is a figure 
	with	shape	p.polygon_shape
	A polygon p at a point base is drawn as follows:
		drawPicture	[setPenPos base:map draw shape]++[drawToPoint base]
*/
instance Drawables Polygon
instance Fillables Polygon
/*	None of these functions change the pen position.
*/
